% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/utils.R
\name{taxonomy_to_tbl_graph}
\alias{taxonomy_to_tbl_graph}
\title{Convert Multi-Column Taxonomy to a Single-Root tbl_graph (with node metadata)}
\usage{
taxonomy_to_tbl_graph(tax_df, columns = NULL, root_name = "AllCells")
}
\arguments{
\item{tax_df}{A data frame with one row per unique path in the hierarchy.
For example, if your columns are \code{c(\"broad\",\"mid\",\"fine\")}, each
row is a single path from \code{broad -> mid -> fine}.}

\item{columns}{A character vector of column names in \code{tax_df} to use.
They should be ordered from the broadest level (first) to the finest level
(last). If \code{NULL}, the function will use all columns of
\code{tax_df} in their given order.}

\item{root_name}{A character string naming the artificial root node,
inserted above the first hierarchy column. Default is \code{\"AllCells\"}.}
}
\value{
A \code{tbl_graph} object (directed) with a single root node. The node
  data includes extra columns corresponding to each level in \code{columns}.
  If a node corresponds to multiple categories at a given level, these are
  combined with \code{\"|\"}.
}
\description{
This function takes a data frame describing a hierarchical taxonomy across
multiple columns (e.g., broad -> mid -> fine). Each row represents a unique
path through the hierarchy. The function introduces a single root node
(named \code{root_name}) above the first hierarchy column, then constructs
a directed tree in which each level connects to the next.
After building the graph, it appends node-level metadata by looking up
which rows (and columns) in \code{tax_df} contain each node. This allows
you to color or facet by different levels of the taxonomy when using \pkg{ggraph}.
}
\details{
1. The function first builds an edge list
   \enumerate{
     \item \code{Root -> level1} for each row
     \item \code{level1 -> level2}
     \item \ldots
     \item \code{level_{N-1} -> levelN}
   }
   and removes duplicates, creating a single connected tree.

2. It then \emph{annotates each node} with the best-known taxonomy data. For
   a node named \code{x}, we look up all rows of \code{tax_df} where
   \code{x} appears in \code{columns}, gather the distinct values from each
   \code{col}, and store them joined with \code{\"|\"} if more than one
   distinct value is found.

This means if a node is shared among multiple broad categories (uncommon, but
possible), that node's \code{broad} column will contain something like
\code{\"Epithelial|Stromal\"}.
}
\examples{
# Minimal example with a 3-level hierarchy (broad -> mid -> fine)
tax_df_example <- data.frame(
    broad = c("Epithelial", "Epithelial", "Stromal"),
    mid   = c("Alveolar", "Alveolar", "Fibroblast"),
    fine  = c("AlveolarType1", "AlveolarType2", "Fibroblast1"),
    stringsAsFactors = FALSE
)
library(tidygraph)
library(ggraph)
library(ggplot2)

# Build a single-root tree and incorporate node metadata
tbl_g <- taxonomy_to_tbl_graph(
    tax_df_example,
    columns   = c("broad", "mid", "fine"),
    root_name = "AllCells"
)

# Inspect node data (metadata for each node)
as.data.frame(tbl_g, "nodes")

# Visualize with ggraph, coloring by 'broad' level
ggraph(tbl_g, layout = "tree") +
    geom_edge_diagonal() +
    geom_node_point(aes(color = broad), size = 3) +
    geom_node_text(aes(label = name), vjust = 1, hjust = 0.5) +
    theme_minimal() +
    labs(title = "Single-Root Taxonomy Tree")

}
